//--------------------------------------------------------------------------
// @riotuart.h
//
// (c) RIoT Secure AB
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// DEVELOPER WARNING
//
// do not modify the source code in any way - it is coded specifically to
// interface with the RIoTClient for the exchange of messages to and from
// the microcontroller, local modem level services and internet services.
//
// RIoTUART communication is restricted to send up to 255 bytes of data
// at a time (len is defined as a uint8_t within the communication protocol)
// the communication protocol is as follows, the code provided implements
// the receival and sending of data over a UART confirming to the protocol.
//
//   0x02 LEN { DATA } CS 0x03
//
//--------------------------------------------------------------------------

#ifndef RIOTUART_H
#define RIOTUART_H

#if   !defined(RIOT_UART_NATIVE) && !defined(RIOT_UART_SOFTWARE)
#error "RIoTUART - must define either RIOT_SERIAL_NATIVE or RIOT_SERIAL_SERIAL"
#elif  defined(RIOT_UART_NATIVE) && defined(RIOT_UART_SOFTWARE)
#error "RIoTUART - cannot define both RIOT_SERIAL_NATIVE or RIOT_SERIAL_SOFTWARE together"
#endif

#define CORE_SER_MAX_DATA_LEN 256
#define CORE_SER_RX_BUF_SIZE  (CORE_SER_MAX_DATA_LEN + 2)  //          { DATA } CS 0x03 - 2 extra bytes
#define CORE_SER_TX_BUF_SIZE  (CORE_SER_MAX_DATA_LEN + 4)  // 0x02 LEN { DATA } CS 0x03 - 4 extra bytes

//--------------------------------------------------------------------------
// globals
//--------------------------------------------------------------------------

int8_t  g_application_status;

#define APP_STATE_IDLE               0
#define APP_STATE_INIT               1
#define APP_STATE_LOOP               2
#define APP_STATE_QUIT               3

// SerialX is using the standard Serial UART
#ifdef RIOT_UART_NATIVE
#define SerialX Serial
#endif

// SerialX is using a SoftwareSerial UART using custom pins
#ifdef RIOT_UART_SOFTWARE
#define serialXRxPin MOSI
#define serialXTxPin MISO
SoftwareSerial SerialX(serialXRxPin, serialXTxPin);
#endif

typedef void (*RIoTUARTCallback)(uint8_t *buf, size_t len);

class RIoTUART
{
  public:
    void     begin(RIoTUARTCallback callback);
    void     loop();
    bool     queue(uint8_t *buf, size_t ofs, size_t len);
    void     end();

  private:

    // our receive buffer
    bool     rx_start;
    uint8_t  rx_buf[CORE_SER_RX_BUF_SIZE];
    size_t   rx_buf_idx;
    size_t   rx_len;
    uint8_t  rx_cs;

    // our transmit buffer
    uint8_t  tx_buf[CORE_SER_TX_BUF_SIZE];
    size_t   tx_buf_idx;
    size_t   tx_len;
    uint8_t  tx_cs;

    RIoTUARTCallback callback;
};

// there should only be one instance of RIoTUART defined
RIoTUART riotUART;

//--------------------------------------------------------------------------
// implementation
//--------------------------------------------------------------------------

// void
// RIoTUART::begin(RIoTUARTCallback callback)
//
// initialize the connection between the RIoTClient and the application.
// this should be placed in the setup() function of the main sketch
//
// @param callback: function pointer - executed on receival of a data packet
// @return none

void
RIoTUART::begin(RIoTUARTCallback callback)
{
  // initialize the serial connection to RIoT Fusion
#ifdef RIOT_UART_SOFTWARE
  pinMode(serialXRxPin, INPUT);
  pinMode(serialXTxPin, OUTPUT);
#endif
  SerialX.begin(57600);

  // initialize buffers
  memset(this->rx_buf, 0, CORE_SER_RX_BUF_SIZE);
  this->rx_start   = false;
  this->rx_buf_idx = 0;
  this->rx_len     = 0x100; // impossible value

  // initialize buffers - for sending data
  memset(this->tx_buf, 0, CORE_SER_TX_BUF_SIZE);
  this->tx_buf_idx = 2;
  this->tx_buf[0]  = 0x02; // STX
  this->tx_buf[1]  = 0x00; // len (to be provided later)

  // register the callback when data is received
  this->callback = callback;
}

// void
// RIoTUART::loop()
//
// maintain connection with RIoTUART, recieve and send messages from
// to allow communication betwenthe RIoTClient and the application.
// this should be placed in the loop() function of the main sketch
// user derived code in should not block code execution; or the UART
// connection may not be maintained and will prevent exchange of data
//
// @param  none
// @return none

void
RIoTUART::loop()
{
  //
  // RECEIVING - attempt to receive a whole message if possible
  //

  // does any data exist on the serial UART?
  while (SerialX.available())
  {
    // read the character from the UART
    uint8_t c = (uint8_t)SerialX.read();
    switch (c)
    {
      // STX
      case 0x02:
           // if no packet defined yet - great, we may have one
           if (!this->rx_start && (this->rx_len == 0x100))
           {
             this->rx_buf_idx = 0;
             this->rx_start   = true;  // ok; we now have a packet
           }

           // if the packet is started, the 0x02 is part of the packet
           else
           {
             // store the 0x02 in the receive buffer
             goto riotuart_loop_store;
           }
           break;

      // ETX
      case 0x03:
           // packet hasn't started - ignore data until STX received
           if (this->rx_len != 0x100)
           {
             // if the packet is started, the 0x03 may be part of the packet
             if (this->rx_buf_idx < (this->rx_len+1))
             {
               // store the 0x03 in the receive buffer
               goto riotuart_loop_store;
             }
             else

             // if we have received all data - the end of the message
             {
               // calculate the checksum (excluding CS itself)
               this->rx_cs = 0;
               for (size_t i=0; i<this->rx_len; i++)
                 this->rx_cs ^= this->rx_buf[i];

               // does the checksum match (ensure type == same)
               if (this->rx_cs == this->rx_buf[this->rx_len])
               {
#if defined(ENABLE_DEBUG)
                 // debugging purposes
                 dbgUART.println(":: recv");
                 dbgUART.print("length:     ");
                 dbgUART.println(this->rx_len);
                 dbgUART.print("checksum:   0x");
                 c = this->rx_cs;
                 if (c < 16) dbgUART.write('0');
                 dbgUART.println(c, HEX);
                 dbgUART.print("recv (HEX): ");
                 for (size_t i=0;i<this->rx_len; i++)
                 {
                   c = this->rx_buf[i];
                   if (c < 16) dbgUART.write('0');
                   dbgUART.print(c, HEX);
                   if (i != (this->rx_len-1)) dbgUART.write(' ');
                 }
                 dbgUART.println();
                 dbgUART.print("recv (ASC): ");
                 for (size_t i=0; i<this->rx_len; i++)
                 {
                   c  = this->rx_buf[i];
                   if ((c >= 0x20) && (c < 0x7f)) dbgUART.write(c);
                   else                           dbgUART.write('.');
                 }
                 dbgUART.println();
                 dbgUART.flush();
#endif

                 //
                 // MICROCONTROLLER CONTROL
                 //

                 // START
                 if (strncmp((char *)this->rx_buf, "START", 5) == 0)
                 {
                   g_application_status = APP_STATE_INIT;
                 }
                 else

                 // FINISH
                 if (strncmp((char *)this->rx_buf, "FINISH", 6) == 0)
                 {
                   g_application_status = APP_STATE_QUIT;
                 }
                 else

                 //
                 // OTHER MESSAGES - WHILE RUNNING
                 //

                 // we only process messages if the application is running
                 if (g_application_status == APP_STATE_LOOP)
                 {
                   this->callback(this->rx_buf, this->rx_len);
                 }
                 else

                 // not started and receiving messages already - initialize
                 {
                   g_application_status = APP_STATE_INIT;
                 }

                 // prepare for a new message to be received
                 this->rx_start   = false;
                 memset(this->rx_buf, 0, CORE_SER_RX_BUF_SIZE);
                 this->rx_buf_idx = 0;
                 this->rx_len     = 0x100; // impossible value
               }
               else
               {
#if defined(ENABLE_DEBUG)
                 dbgUART.println(":: RIoTUART.recv - message checksum mismatch");
#endif
               }
             }
           }

           // we are done; move along to next message
           goto riotuart_loop_recv_done;

      default:

riotuart_loop_store:;

           // start of packet? first byte is length
           if (this->rx_start)
           {
             this->rx_start = false;
             this->rx_len   = c;
           }
           else

           // packet hasn't started - ignore data
           if (this->rx_len != 0x100)
           {
             // store the value
             this->rx_buf[this->rx_buf_idx++] = c;

             // have we received too many bytes? malformed packet or bogus?
             if (this->rx_buf_idx > (this->rx_len+2))
             {
#if defined(ENABLE_DEBUG)
               // debugging purposes
               dbgUART.println(":: RIoTUART.recv - rx buffer overflow");
#endif

               // prepare for a new message to be received
               this->rx_start   = false;
               memset(this->rx_buf, 0, CORE_SER_RX_BUF_SIZE);
               this->rx_buf_idx = 0;
               this->rx_len     = 0x100; // impossible value

               // we are done; move along to next message
               goto riotuart_loop_recv_done;
             }
           }
           break;
    }
  }

riotuart_loop_recv_done:;

  //
  // SENDING - if the developer has queued up any messages, send them
  //

  // do we have anything to send?
  if ((this->tx_buf_idx > 2) && (this->tx_buf_idx < 257))
  {
    // we now know the length of our packet
    this->tx_len    = this->tx_buf_idx-2;
    this->tx_buf[1] = this->tx_len;

    // calculate the checksum of the entire message
    this->tx_cs = 0;
    for (size_t i=0; i<this->tx_len; i++)
      this->tx_cs ^= this->tx_buf[i+2];

    // and now we know the checksum
    this->tx_buf[this->tx_len+2] = this->tx_cs;
    this->tx_buf[this->tx_len+3] = 0x03; // ETX

#if defined(ENABLE_DEBUG)
    uint8_t c;

    dbgUART.println(F(":: send"));
    dbgUART.print(F("length:     "));
    dbgUART.println(this->tx_len);
    dbgUART.print(F("checksum:   0x"));
    c = this->tx_cs;
    if (c < 16) dbgUART.print(F("0"));
    dbgUART.println(c, HEX);
    dbgUART.print(F("send (HEX): "));
    for (size_t i=0;i<this->tx_len; i++)
    {
      c = this->tx_buf[i+2];
      if (c < 16) dbgUART.write('0');
      dbgUART.print(c, HEX);
      if (i != (this->tx_len-1)) dbgUART.write(' ');
    }
    dbgUART.println();
    dbgUART.print(F("send (ASC): "));
    for (size_t i=0; i<this->tx_len; i++)
    {
      c = this->tx_buf[i+2];
      if ((c >= 0x20) && (c < 0x7f)) dbgUART.write(c);
      else                           dbgUART.write('.');
    }
    dbgUART.println();
#endif

    // 0x02 LEN { DATA } CS 0x03
    SerialX.write(this->tx_buf, this->tx_len+4);
    SerialX.flush();

    // initialize buffers - for sending data
    memset(this->tx_buf, 0, CORE_SER_TX_BUF_SIZE);
    this->tx_buf_idx = 2;
    this->tx_buf[0]  = 0x02; // STX
    this->tx_buf[1]  = 0x00; // len (to be provided later)
  }
}

// void
// RIoTUART::queue(uint8_t *buf, size_t ofs, size_t len)
//
// queue a message into the RIoTUART that will be sent during the
// subequence maintenance handler of the RIoTUART connection.
//
// @param  buf - a pointer to the buffer to send
// @param  ofs - the offet within the buffer to retrieve bytes
// @param  len - the number of bytes to queue
// @return true on success, false if buffer is full
//
// EXAMPLE USAGE:
//
// {
//   if (riotUART.queue(rx, len)) // success
//   else { }                     // failure, buffer full
// }

bool
RIoTUART::queue(uint8_t *buf, size_t ofs, size_t len)
{
  bool success = false;

  // do we have room for the message in the tx buffer?
  if ((this->tx_buf_idx + len) < (CORE_SER_TX_BUF_SIZE-4))
  {
    // insert the message into the queue
    memcpy(&this->tx_buf[this->tx_buf_idx], &buf[ofs], len);
    this->tx_buf_idx += len;

    // inform the developer the message was queued successfully
    success = true;
  }

  return success;
}

// void
// RIoTUART::end()
//
// terminate the connection between the RIoTClient and the application.
//
// @param  none
// @return none

void
RIoTUART::end()
{
  // terminate the serial connection to RIoT Fusion
  SerialX.end();
#ifdef RIOT_UART_SOFTWARE
  pinMode(serialXRxPin, INPUT);
  pinMode(serialXTxPin, INPUT);
#endif
}

#endif

//--------------------------------------------------------------------------
