/*
File: application.h

Handle communication with the customer cloud.

A counter value is used as example data.
*/

#ifndef APPLICATION_H
#define APPLICATION_H

// LED matrix library
#include "Arduino_LED_Matrix.h"

// Create matrix object
ArduinoLEDMatrix g_led_matrix;

// LED matrix screens
const uint32_t g_matrix_send[]  = { 0xe968d5eb, 0x5295e960, 0x0, 66 };
const uint32_t g_matrix_ok[]    = { 0xf4895096, 0x950f480, 0x0, 66 };
const uint32_t g_matrix_er[]    = { 0xf70848f7, 0x8850f480, 0x0, 66 };
//const uint32_t g_matrix_blank[] = { 0x0, 0x0, 0x0, 66 };

// Constants
#define       MESSAGE_INTERVAL         15000 // millisecond interval
#define       MESSAGE_PACKET_SIZE      5     // size of counter packet
#define       GPS_MESSAGE_PACKET_SIZE  19    // size of predefined GPS packet

// Global variables
unsigned long g_timer;
unsigned int  g_counter;
uint8_t       g_message[MESSAGE_PACKET_SIZE];
char          g_inbuf[64];

void send_counter()
{
  g_led_matrix.loadFrame(g_matrix_send);

  uint16_t value = (uint16_t) g_counter;

  g_message[0] = '*'; // * = high priority,  # = normal priority
  g_message[1] = 'c'; // packet id
  g_message[2] = 2;   // data length in bytes
  g_message[3] = 0xFF & (value >> 8);
  g_message[4] = 0xFF & (value);

  // Queue message for sending to server
  riotUART.queue(g_message, 0, MESSAGE_PACKET_SIZE);

  #ifdef ENABLE_DEBUG
    dbgUART.print(F("SEND COUNTER: "));
    dbgUART.println(value);
  #endif
}

void application_init()
{
  #ifdef ENABLE_DEBUG
    dbgUART.println(F("APPLICATION INIT"));
    dbgUART.print(F("MESSAGE INTERVAL: "));
    dbgUART.println(MESSAGE_INTERVAL);
  #endif

  // Initialize LED matrix
  g_led_matrix.begin();
  //g_led_matrix.loadFrame(g_matrix_blank);

  // Set initial counter
  g_counter = 0;

  // Set timestamp for send timer
  g_timer = millis();
}

void application_loop()
{
  // Send counter to server on MESSAGE_INTERVAL
  unsigned long now = millis();
  if ((g_timer + MESSAGE_INTERVAL) < now)
  {
    // Increment and send counter value
    g_counter += 1;
    send_counter();

    // Update interval timestamp
    g_timer = millis();
  }
}

// Callback function that handles messages from the Core Firmware
void application_riotuart_callback(uint8_t *rx, size_t len)
{
  // Check packet message format
  if (rx[0] == '#') // # means incoming data packet
  {
    switch (rx[1])
    {
      case '+':
        // Check if valid GPS packet and handle data
        if (len == GPS_MESSAGE_PACKET_SIZE)
        {
          // Process GPS data and send to cloud server
          // (optional, not handled in this example)
        }
        break;

      default:
        break;
    }
  }
  else
  // Handle string data sent from the cloud server
  {
    // Save string data to a buffer
    int length = len < sizeof(g_inbuf) ? len : sizeof(g_inbuf);
    memcpy(g_inbuf, rx, length);
    g_inbuf[length] = 0;

    // Update LED matrix display
    if (0 == strcmp(g_inbuf, "OK"))
    {
      g_led_matrix.loadFrame(g_matrix_ok);
    }
    else
    {
      g_led_matrix.loadFrame(g_matrix_er);
    }

    #ifdef ENABLE_DEBUG
      dbgUART.print(F("INCOMING DATA LENGTH: "));
      dbgUART.println(len);
      dbgUART.println(g_inbuf);
    #endif
  }
}

void application_quit()
{
  // Shutdown sub-systems as needed
}

#endif
