<?php
/*
File: device_time_to_update.php

Print a progress bar that shows the time left to the next expected
device connection.

This script can be used as an inspiration for a device monitoring tool.

Terminate the script by pressing CTRL-C

Usage:

    php device_time_to_update.php LOGINFILE DEVICENAME

Example:

    php device_time_to_update.php mylogin.json "Test Device"

The script could easily be modified to take a device id as
parameter instead of the device name.
*/

require_once __DIR__ . "/../../tools/lib/rest_api_lib.php";
require_once __DIR__ . "/../../tools/lib/device_lib.php";
require_once __DIR__ . "/../../tools/lib/ansi_codes.php";

main($argc, $argv);

function main($argc, $argv)
{
  // Set command params and read login config file
  if ($argc == 3)
  {
    $login_config_file = $argv[1]; // absolute or relative path to config file
    $device_name       = $argv[2]; // name of device

    // Read login file
    $config_path = get_absolute_path($login_config_file);
    rest_api_read_login_config($config_path);
  }
  else
  {
    echo "Usage:" . PHP_EOL;
    echo "php device_time_to_update.php LOGINFILE DEVICENAME" . PHP_EOL;
    echo "Example:" . PHP_EOL;
    echo "php device_time_to_update.php mylogin.json \"Test Device\"" . PHP_EOL;
    die();
  }

  echo "Monitoring time until next connection (quit with ctrl-c)" . PHP_EOL;

  // Get tenant id
  $tenant_id = rest_api_tenant_get_id();

  // Get device object
  $device = device_find($tenant_id, $device_name);
  if (false === $device)
  {
    echo "ERROR: Could not find device with name: " . $device_name . PHP_EOL;
    die();
  }

  $device_id = $device->id;

  $response = rest_api_get_expand("/global");
  $global_obj = json_decode($response);
  $interval = $global_obj->keys->sync_period_low;
  $time_left_prev = $interval;

  //$TIME_LAST_SEEN = time() - 30; // debug code

  while (true)
  {
    //$time_last_seen = $TIME_LAST_SEEN; // debug code
    $time_last_seen = device_time_last_connect($tenant_id, $device->id);
    $time_estimated_next = $time_last_seen + $interval;
    $time_left = $time_estimated_next - time();
    //$offline = $time_left < 0;

    if ($time_left > $time_left_prev)
    {
      // Time has elapsed
      // OLD_CODE: echo "!" . PHP_EOL;
    }
    else
    if ($time_left < 0) // offline
    {
      // Lost contact with device
      CursorColumn(1);
      BackgroundWhite();
      TextGray();
      echo "Waiting";
      AnsiReset();

      $TIME_LAST_SEEN = time() - 30;
    }
    else
    {
      // Show time left as a progress bar
      $time_elapsed = 60 - $time_left;
      CursorColumn(1);
      BackgroundWhite();
      echo str_repeat(" ", $time_elapsed);
      AnsiReset();
      echo str_repeat(" ", $time_left);
      echo VBAR;
      CursorColumn(1);
    }

    $time_left_prev = $time_left;

    sleep(1);
  }
}
