<?php
/*
File: hardware_generate_html.php

This script shows an example of how HTML can be genrated from
REST API calls.

The script generates HTML pages with supported modems
and microcontrollers.

Generated files are saved in the current working directory.

Usage:

    php hardware_generate_html.php LOGINFILE

Example:

    php hardware_generate_html.php mylogin.json
*/

require_once __DIR__ . "/../../tools/lib/rest_api_lib.php";

// Call main function
main($argc, $argv);

function main($argc, $argv)
{
  // Read config file with user credentials
  if ($argc == 2)
  {
    $login_config_file = $argv[1];
    $config_path = get_absolute_path($login_config_file);
    rest_api_read_login_config($config_path);
  }
  else
  {
    echo "USAGE:" . PHP_EOL;
    echo "php hardware_generate_html.php LOGINFILE" . PHP_EOL;
    echo "EXAMPLE:" . PHP_EOL;
    echo "php hardware_generate_html.php mylogin.json" . PHP_EOL;
    die();
  }

  // Destination directory
  $dir = getcwd() . "/hardware";
  @mkdir($dir);

  // Generate microcontroller data
  $microcontrollers = microcontroller_get();
  microcontroller_generate_html($microcontrollers, $dir);
  download_images($microcontrollers, $dir);

  // Generate modem data
  $modems = modem_get();
  modem_generate_html($modems, $dir);
  download_images($modems, $dir);

  echo "HTML file and images created in folder:" . PHP_EOL;
  echo $dir . PHP_EOL;
}

function microcontroller_get()
{
  // Get all microcontrollers (returns JSON array of objects)
  $response = rest_api_get_expand("/microcontroller");

  // Convert JSON to PHP array
  $microcontrollers = json_decode($response);

  return $microcontrollers;
}

function modem_get()
{
  // Get all modems (returns JSON array of objects)
  $response = rest_api_get_expand("/modem");

  // Convert JSON to PHP array
  $modems = json_decode($response);

  return $modems;
}

function microcontroller_generate_html($microcontrollers, $dir)
{
  echo "Generate microcontroller HTML" . PHP_EOL;

  $html_file = $dir . "/microcontrollers.html";

  $html = "<html><body>" . PHP_EOL;

  $html .= "<style>.hardware-image { width: 100px; height: auto; }</style>" . PHP_EOL;

  $html .= "<table>" . PHP_EOL;
  $html .= "<tr><th>Image</th><th>Identifier</th><th>Name</th><th>Architechture</th><th>Manufacturer</th><th>RAM</th><th>Flash</th><th>REST&nbsp;ID</th></tr>" . PHP_EOL;

  foreach ($microcontrollers as $microcontroller)
  {
    echo "Download microcontroller: " . $microcontroller->keys->name . PHP_EOL;

    $image_id = $microcontroller->keys->image;
    $image_extension = image_get_extension($image_id);
    $image_file_name = $image_id . "." . $image_extension;

    $html .= "<tr>" . PHP_EOL;
    $html .=
    "<td><img class=\"hardware-image\" src=\"images/" . $image_file_name . "\"></td>" .
      "<td>" . $microcontroller->keys->identifier . "</td>" .
      "<td>" . $microcontroller->keys->name . "</td>" .
      "<td>" . $microcontroller->keys->arch . "</td>" .
      "<td>" . $microcontroller->keys->manufacturer . "</td>" .
      "<td>" . $microcontroller->keys->ram . "</td>" .
      "<td>" . $microcontroller->keys->flash . "</td>" .
      "<td>" . $microcontroller->id . "</td>" .
      PHP_EOL;
      //"<td>" . $microcontroller->keys->programmer . "</td>" .
      //"<td>" . $microcontroller->keys->eeprom . "</td>" . PHP_EOL;
    $html .= "</tr>" . PHP_EOL;
  }

  $html .= "</table>" . PHP_EOL;
  $html .= "</body></html>" . PHP_EOL;

  file_put_contents($html_file, $html);
}

function modem_generate_html($modems, $dir)
{
  echo "Generate modem HTML" . PHP_EOL;

  $html_file = $dir . "/modems.html";

  $html = "<html><body>" . PHP_EOL;

  $html .= "<style>.hardware-image { width: 100px; height: auto; }</style>" . PHP_EOL;

  $html .= "<table>" . PHP_EOL;
  $html .= "<tr><th>Image</th><th>Identifier</th><th>Name</th><th>Network</th><th>Manufacturer</th><th>Architecture</th><th>RAM</th><th>Flash</th><th>REST&nbsp;ID</th></tr>" . PHP_EOL;

  foreach ($modems as $modem)
  {
    echo "Download modem: " . $modem->keys->name . PHP_EOL;

    $image_id = $modem->keys->image;
    $image_extension = image_get_extension($image_id);
    $image_file_name = $image_id . "." . $image_extension;

    $html .= "<tr>" . PHP_EOL;
    $html .=
      "<td><img class=\"hardware-image\" src=\"images/" . $image_file_name . "\"></td>" .
      "<td>" . $modem->keys->identifier . "</td>" .
      "<td>" . $modem->keys->name . "</td>" .
      "<td>" . $modem->keys->network . "</td>" .
      "<td>" . $modem->keys->manufacturer . "</td>" .
      "<td>" . $modem->keys->arch . "</td>" .
      "<td>" . $modem->keys->ram . "</td>" .
      "<td>" . $modem->keys->flash . "</td>" .
      "<td>" . $modem->id . "</td>" .
      PHP_EOL;
    $html .= "</tr>" . PHP_EOL;
  }

  $html .= "</table>" . PHP_EOL;
  $html .= "</body></html>" . PHP_EOL;

  file_put_contents($html_file, $html);
}

// Download images for microcontroller or modem
function download_images($objects, $dir)
{
  $image_dir = $dir . "/images";
  @mkdir($image_dir);

  foreach ($objects as $object)
  {
    $image_id = $object->keys->image;
    echo "Download image: " . $image_id . PHP_EOL;
    image_download($image_id, $image_dir);
  }
}

function image_download($image_id, $dir_path)
{
  // Get file extension
  $extension = image_get_extension($image_id);

  // Download image data
  $image_data = rest_api_get("/image/{$image_id}/download");

  // Save image
  $image_path = "{$dir_path}/{$image_id}.{$extension}";
  $file = fopen($image_path, "wb");
  if ($file)
  {
    fwrite($file, $image_data);
    fclose($file);
  }
}

function image_get_extension($image_id)
{
  // Get mime type and return file extension
  $response = rest_api_get_expand("/image/{$image_id}");
  $data = json_decode($response);
  $mime_type = $data->info->mime;
  $extension = ("image/jpeg" === $mime_type) ? "jpg" : "png";
  return $extension;
}

// It is considered best practice to not close the <?php tag
// in a PHP-only file.