<?php
/*
File: tenant_update.php

This example uses a PUT request to update the "website" key
of the current tenant.

Usage:

    php tenant_update.php LOGINFILE

Example:

    php tenant_update.php mylogin.json
*/

require_once __DIR__ . "/../../tools/lib/rest_api_lib.php";

// Helper function
function tenant_update_website($tenant_id, $website)
{
  // Server expects JSON encoding with JSON_UNESCAPED_SLASHES
  $json_data = json_encode($website, JSON_UNESCAPED_SLASHES);
  $response = rest_api_put(
    "/tenant/{$tenant_id}/keys/website",
    $json_data,
    "application/json");

  // Check response code
  if (202 !== rest_api_get_http_response_code())
  {
    echo "ERROR: COULD NOT UPDATE WEBSITE KEY\n";
    die();
  }
}

// Read config file with user credentials
if ($argc == 2)
{
  $login_config_file = $argv[1];
  $config_path = get_absolute_path($login_config_file);
  rest_api_read_login_config($config_path);
}
else
{
  echo "USAGE:" . PHP_EOL;
  echo "php tenant_update.php LOGINFILE" . PHP_EOL;
  echo "EXAMPLE:" . PHP_EOL;
  echo "php tenant_update.php mylogin.json" . PHP_EOL;
  die();
}

// Get tenant id
$tenant_id = rest_api_tenant_get_id();

// Get current tenant website address
$response = rest_api_get("/tenant/{$tenant_id}/keys/website");
$website_orig = json_decode($response);

echo "--------------------------------------" . PHP_EOL;
echo "TENANT UPDATE EXAMPLE" . PHP_EOL;
echo "--------------------------------------" . PHP_EOL;
echo "ORIGINAL WEBSITE KEY : {$website_orig}" . PHP_EOL;

// Update website
$website = "https://updatedwebsite.com";
tenant_update_website($tenant_id, $website);

// Show updated value
$response = rest_api_get("/tenant/{$tenant_id}/keys/website");
$website_updated = json_decode($response);
echo "UPDATED  WEBSITE KEY : {$website_updated}" . PHP_EOL;

// Restore original value
tenant_update_website($tenant_id, $website_orig);

// Show restored value
$response = rest_api_get("/tenant/{$tenant_id}/keys/website");
$website_restored = json_decode($response);
echo "RESTORED WEBSITE KEY : {$website_restored}" . PHP_EOL;

// It is considered best practice to not close the <?php tag
// in a PHP-only file.