<?php
/*
File: instdl.php
Copyright (c) 2023-2024, RIoT Secure AB

WARNING: EXPERIENCED DEVELOPERS ONLY

Download RIoTInstaller for a single device or
for multiple devices.

Create a folder with an install sketch named
RIoTInstaller_<DEVICEID>.ino that installs the
RIoT Core Firmware on the device and enrols it.

The folder will be created in the current working
directory.

Usage:

    php instdl.php LOGINFILE DEVICEID

Example:

    php instdl.php mylogin.json 42

The above command will download a RIoTInstaller sketch
for the device with id 42. The command will be create a
folder named RIoTInstaller_42, which will contain the
sketch RIoTInstaller_42.ino

To download installers for multiple devices, specify two
devices ids:

    php instdl.php LOGINFILE DEVICEID DEVICEID_LAST

Example:

    php instdl.php mylogin.json 42 45

The above command will create folders with RIoTInstaller
sketches for the devices starting with id 42, up to and
including id 46. In this example, the following folders
will be created:

    RIoTInstaller_42
    RIoTInstaller_43
    RIoTInstaller_44
    RIoTInstaller_45

The folders will be created in the current working
directory.
*/

require_once __DIR__ . "/../lib/rest_api_lib.php";
require_once __DIR__ . "/../lib/device_lib.php";

// Call main function
main($argc, $argv);

function main($argc, $argv)
{
  // Read config file with user credentials
  if ($argc == 3 || $argc == 4)
  {
    // Set parameters
    $login_config_file = $argv[1];
    $device_id = $argv[2];

    // Read login file
    $config_path = get_absolute_path($login_config_file);
    rest_api_read_login_config($config_path);

    if ($argc == 3)
    {
      download_single_installer($device_id);
    }
    else
    {
      $device_id_last = $argv[3];
      download_multiple_installers($device_id, $device_id_last);
    }
  }
  else
  {
    echo "USAGE:" . PHP_EOL;
    echo "php instdl.php LOGINFILE DEVICEID [DEVICEID_LAST]" . PHP_EOL;
    echo "EXAMPLE (create RIoTInstaller sketch for device with id 42):" . PHP_EOL;
    echo "php instdl.php mylogin.json 42" . PHP_EOL;
    die();
  }
}

function download_single_installer($device_id)
{
  download_installer_sketch($device_id);

  echo "Installer sketch created in folder:" . PHP_EOL;
  echo getcwd() . "/RIoTInstaller_{$device_id}" . PHP_EOL;
  echo "Instructions:" . PHP_EOL;
  echo "Connect the modem to the computer and open the" . PHP_EOL;
  echo "sketch RIoTInstaller_{$device_id}.ino in the Arduino IDE." . PHP_EOL;
  echo "Run the sketch to install the core firmware." . PHP_EOL;
  echo "When the sketch has finished, press the reset" . PHP_EOL;
  echo "button on the modem to reboot and enroll the device." . PHP_EOL;
}

function download_multiple_installers($device_id, $device_id_last)
{
  for ($id = $device_id; $id <= $device_id_last; $id++)
  {
    echo "Download RIoTInstaller_" . $id . PHP_EOL;

    download_installer_sketch($id);

    sleep(1);
  }

  echo "Installers created in folder:" . PHP_EOL;
  echo getcwd() . PHP_EOL;
  echo "Instructions:" . PHP_EOL;
  echo "Connect the modem to the computer and open the" . PHP_EOL;
  echo "RIoTInstaller ino sketch in the Arduino IDE." . PHP_EOL;
  echo "Run the sketch to install the core firmware." . PHP_EOL;
  echo "When the sketch has finished, press the reset" . PHP_EOL;
  echo "button on the modem to reboot and enroll the device." . PHP_EOL;
}

function download_installer_sketch($device_id)
{
  // Download zip file with installer sketch

  // Get tenant id
  $tenant_id = rest_api_tenant_get_id();

  // Set sketch base name
  $sketch_base_name = "RIoTInstaller_{$device_id}";

  // Download installer sketch
  $zip_data = device_download_installer_sketch($tenant_id, $device_id);
  if (200 !== rest_api_get_http_response_code())
  {
    exit("[ERROR] cannot download installer sketch for device id: {$device_id}");
  }

  // Save sketch file
  $zip_file = getcwd() . "/{$sketch_base_name}.zip";
  file_put_contents($zip_file, $zip_data);

  // Extract zipped files
  $zip = new ZipArchive();
  $result = $zip->open($zip_file);
  if (TRUE === $result)
  {
    $zip->extractTo(getcwd());
    $zip->close();
  }
  else
  {
    echo "[ERROR] Could not extract files from zip archive" . PHP_EOL;
  }

  unlink($zip_file);
}
