<?php
/*
Library of functions for managing devices and firmwares.

NOT for production use. Intended for educational purposes.
*/

require_once __DIR__ . "/rest_api_lib.php";

// --------------------------------------------------------------
// DEVICE FUNCTIONS
// --------------------------------------------------------------

/*
Example of call to device_create():

$network_settings           = new stdClass();
$network_settings->SSID     = "MY WIFI NAME";
$network_settings->password = "MY PASSWORD";

$gps_settings               = new stdClass();
$gps_settings->latitude     = "59.916675";
$gps_settings->longitude    = "18.950068";

$response = device_create(
    $tenant_id,
    "Test Device",                      // device_name,
    "MKR WiFi 1010",                    // modem_name,
    "RIoT Fusion Shield - Arduino UNO", // hardware_interface_name,
    "1_0_0_16-riot-MKRWIFI1010",        // firmware_core_name,
    "WiFi",                             // network_type,
    $network_settings,
    $gps_settings);
*/

// Create a new device object
// Return query response
function device_create(
  $tenant_id,
  $device_name,
  $modem_name,
  $hardware_interface_name,
  $firmware_core_name,
  $network_type,
  $network_settings = null,
  $static_gps_settings = null)
{
  // Find modem identifier
  $modem = modem_find($modem_name);
  $modem_identifier = $modem->keys->identifier;

  // Create device object and set fields
  $device_obj = new stdClass();
  $device_obj->keys = new stdClass();
  $device_obj->keys->name = $device_name;
  $device_obj->keys->modem = $modem_name;
  $device_obj->keys->identifier = $modem_identifier;
  $device_obj->keys->interface = $hardware_interface_name;
  $device_obj->keys->firmware_core = $firmware_core_name;
  $device_obj->keys->integration = false;
  $device_obj->keys->network = $network_type;

  // Set network fields ("Ethernet" network type has no settings)
  if ($network_type !== "Ethernet")
  {
    $device_obj->keys->settings = $network_settings;
  }

  // Set static GPS fields
  if (null !== $static_gps_settings)
  {
    $device_obj->keys->gps = $static_gps_settings;
  }

  $data = json_encode($device_obj);

  $response = rest_api_post(
    "/tenant/{$tenant_id}/device",
    $data,
    "application/json");

  return $response;
}

// Delete a device
// Return query response
function device_delete($tenant_id, $device_id)
{
  $response = rest_api_delete("/tenant/{$tenant_id}/device/{$device_id}");
  return $response;
}

// Download an install file for use with an SD card
// Return binary data
function device_download_sdupdate($tenant_id, $device_id)
{
  $response = rest_api_get("/tenant/{$tenant_id}/device/{$device_id}/download");
  return $response;
}

// Download an install sketch for use with the Arduino IDE
// Return binary data with a zip file
function device_download_installer_sketch($tenant_id, $device_id)
{
  $response = rest_api_get("/tenant/{$tenant_id}/device/{$device_id}/download_installer");
  return $response;
}

// Get device by id
// Return device object on success, or false on error
function device_get($tenant_id, $device_id)
{
  $response = rest_api_get_expand("/tenant/{$tenant_id}/device/{$device_id}");
  if (200 === rest_api_get_http_response_code())
  {
    return json_decode($response);
  }
  else
  {
    return false;
  }
}

// Return the device object with the given name (assumes device has unique name)
// Return false if not found
function device_find($tenant_id, $device_name)
{
  // Get all devices
  $response_json = rest_api_get_expand("/tenant/{$tenant_id}/device");
  if (200 !== rest_api_get_http_response_code())
  {
    return false;
  }

  $device_array = json_decode($response_json);

  foreach ($device_array as $device)
  {
    if ($device_name == $device->keys->name)
    {
      // Found device - return device object
      return $device;
    }
  }

  // Not found
  return false;
}

// $device_obj is a PHP object; it can be a partial
// device object that contains the keys to be updated
// Return query response on success, false on error
function device_update($tenant_id, $device_id, $device_obj)
{
  $data = json_encode($device_obj);

  $response = rest_api_put(
    "/tenant/{$tenant_id}/device/{$device_id}",
    $data,
    "application/json"
  );

  if (202 === rest_api_get_http_response_code())
  {
    return $response;
  }
  else
  {
    return false;
  }
}

// $device_obj is a PHP object; it can be a partial
// device object that contains the keys to be updated
// Return query response on success, false on error
function device_update_firmware_appl($tenant_id, $device_obj, $firmware_name, $microcontroller_index = 0)
{
  // Update firmware of the microcontroller with the given index
  $microcontroller_array = $device_obj->keys->microcontroller;
  $microcontroller_obj = $microcontroller_array[$microcontroller_index];
  $microcontroller_obj->firmware_appl = $firmware_name;

  // Create device update object that contains only the microcontroller array
  $device_new = new stdClass();
  $device_new->keys = new stdClass();
  $device_new->keys->microcontroller = $microcontroller_array;

  // For debugging
  //var_dump($device_new);

  // Update device
  $device_id = $device_obj->id;
  $response = device_update($tenant_id, $device_id, $device_new);

  return $response;
}

function device_time_last_connect($tenant_id, $device_id)
{
  $response_json = rest_api_get_expand("/tenant/{$tenant_id}/device/{$device_id}");
  $device = json_decode($response_json);
  return $device->info->ts_comms;
}

// --------------------------------------------------------------
// FIRMWARE FUNCTIONS
// --------------------------------------------------------------

// Create application firmware
// Return false on error
function firmware_appl_create(
  $tenant_id,
  $firmware_name,
  $firmware_microcontroller,
  $firmware_version,
  $firmware_description,
  $firmware_absolute_path)
{
  // Get microcontroller identifier
  $microcontroller = microcontroller_find($firmware_microcontroller);
  if (false === $microcontroller)
  {
    echo "[ERROR] firmware_appl_create: microcontroller not found" . PHP_EOL;
    return false;
  }
  $microcontroller_identifier = $microcontroller->keys->identifier;

  // Create handle to firmware file
  $firmware_file_handle = curl_file_create($firmware_absolute_path, "application/octet-stream");

  // Set POST data
  $post_data =
  [
    "name"            => $firmware_name,
    "microcontroller" => $firmware_microcontroller,
    "identifier"      => $microcontroller_identifier,
    "version"         => $firmware_version,
    "description"     => $firmware_description,
    "asset"           => $firmware_file_handle
  ];

  // Make the POST request to create the firmware object
  $response = rest_api_post(
    "/tenant/" . $tenant_id. "/firmware_appl",
    $post_data,
    "multipart/form-data");

  if (201 !== rest_api_get_http_response_code())
  {
    echo "[ERROR] firmware_appl_create: POST failed" . PHP_EOL;
    echo $response . PHP_EOL;
    return false;
  }
  else
  {
    return $response;
  }
}

// Delete application firmware
// Return true on success, false on error
function firmware_appl_delete($tenant_id, $firmware_name)
{
  // Make request to delete the firmware object
  rest_api_delete("/tenant/{$tenant_id}/firmware_appl/{$firmware_name}");
  if (204 === rest_api_get_http_response_code())
  {
    return true;
  }
  else
  {
    return false;
  }
}

// Get application firmware
// Return false on error or if firmware not found
function firmware_appl_get($tenant_id, $firmware_name)
{
  // Get firmware object
  $response = rest_api_get_expand(
    "/tenant/{$tenant_id}/firmware_appl/{$firmware_name}");
  if (200 === rest_api_get_http_response_code())
  {
    $firmware_obj = json_decode($response);
    return $firmware_obj;
  }
  else
  {
    return false;
  }
}

// --------------------------------------------------------------
// MODEM FUNCTIONS
// --------------------------------------------------------------

// Return the modem with the given name
// Return false if not found
function modem_find($modem_name)
{
  $response = rest_api_get_expand("/modem");
  $modem_array = json_decode($response);
  foreach ($modem_array as $modem)
  {
    if ($modem->keys->name === $modem_name)
    {
      return $modem;
    }
  }

  // Not found
  return false;
}

// --------------------------------------------------------------
// MICROCONTROLLER FUNCTIONS
// --------------------------------------------------------------

// Return the microcontroller with the given name
// Return false if not found
function microcontroller_find($microcontroller_name)
{
  $response = rest_api_get_expand("/microcontroller");
  $microcontroller_array = json_decode($response);
  foreach ($microcontroller_array as $microcontroller)
  {
    if ($microcontroller->keys->name === $microcontroller_name)
    {
      return $microcontroller;
    }
  }

  // Not found
  return false;
}

// It is considered best practice to not close the <?php tag
// in a PHP-only file.