<?php
/*
Library for learning the OASIS REST API.

NOT for production use. Intended for educational purposes.

Error handling and other aspects of the library should be
improved for production use.
*/

require_once __DIR__ . "/authorization_lib.php";

// --------------------------------------------------------------
// GLOBALS AND CONSTANTS
// --------------------------------------------------------------

$_SERVER_URL          = null;
$_AUTH_METHOD         = null;
$_USERNAME            = null;
$_PASSHASH            = null;

$_CURL_ERRNO          = 0;
$_CURL_ERROR          = null;
$_HTTP_RESPONSE_CODE  = null;

// Constants for query options
define("OPT_NONE",           1);  // No options set
define("OPT_EXPAND",         2);  // Set the the expand parameter
define("OPT_HEADER_INCLUDE", 4);  // Include header and body in response
define("OPT_HEADER_ONLY",    8);  // Include header only in response
// Default is to include only the body in the response

// --------------------------------------------------------------
// CONFIG FUNCTIONS
// --------------------------------------------------------------

// Set the server URL
function rest_api_set_server_url($server_url)
{
  global $_SERVER_URL;

  $_SERVER_URL = $server_url;
}

// Set user credentials used for authentication
// Note this function only accepts a password hash
function rest_api_set_user_credentials(
  $username,
  $passhash,
  $auth_method = "oasis")
{
  global $_AUTH_METHOD;
  global $_USERNAME;
  global $_PASSHASH;

  $_AUTH_METHOD = $auth_method;
  $_USERNAME    = $username;
  $_PASSHASH    = $passhash;
}

// Read and pars the specified JSON file
function rest_api_read_config_file($path)
{
  $json = file_get_contents($path);
  if (false === $json)
  {
    echo "[rest_api_lib.php] Fatal Error: Could not read JSON file: {$path}".PHP_EOL;
    die();
  }

  // TODO: Error check decode
  $obj = json_decode($json);

  return $obj;
}

// Read the specified config file and set user credentials
function rest_api_read_login_config($path)
{
  $obj = rest_api_read_config_file($path);

  rest_api_set_server_url($obj->server_url);

  rest_api_set_user_credentials(
    $obj->username,
    $obj->passhash,
    $obj->auth_method);
}

// --------------------------------------------------------------
// UTILITY FUNCTIONS
// --------------------------------------------------------------

// Return absolute path based on relative or absolute path name
function get_absolute_path($file_name)
{
  // Check if absolute path (begins with "/")
  if ($file_name[0] === "/")
  {
    // Absolute path - use as is
    return $file_name;
  }
  else
  {
    // Assume relative path
    return getcwd() . "/" . $file_name;
  }
}

// --------------------------------------------------------------
// QUERY FUNCTIONS
// --------------------------------------------------------------

// Generic REST API request function
function rest_api_request(
  $request_method,
  $request_uri,
  $options = OPT_NONE,
  $data = null,
  $content_type = null)
{
  $curl = rest_api_create_query($request_method, $request_uri, $options, $data, $content_type);
  $response = rest_api_curl_exec($curl);
  curl_close($curl);

  return $response;
}

// Make a GET request
function rest_api_get($request_uri)
{
  return rest_api_request("GET", $request_uri);
}

// Make a GET request with the expand query parameter set
function rest_api_get_expand($request_uri)
{
  return rest_api_request("GET", $request_uri, OPT_EXPAND);
}

// Make a POST request
function rest_api_post($request_uri, $data, $content_type)
{
  return rest_api_request("POST", $request_uri, OPT_NONE, $data, $content_type);
}

// Make a PUT request
function rest_api_put($request_uri, $data, $content_type)
{
  return rest_api_request("PUT", $request_uri, OPT_NONE, $data, $content_type);
}

// Make a DELETE request
function rest_api_delete($request_uri)
{
  return rest_api_request("DELETE", $request_uri);
}

// Make a OPTIONS request
function rest_api_options($request_uri)
{
  return rest_api_request("OPTIONS", $request_uri);
}

// Make an OPTIONS request to get the response HEADER only
function rest_api_options_header($request_uri)
{
  return rest_api_request("OPTIONS", $request_uri, OPT_HEADER_ONLY);
}

// Make a OPTIONS preflight request (only header returned)
function rest_api_options_preflight($request_uri, $request_method, $request_headers)
{
  $request_method = "OPTIONS";

  $url = rest_api_url($request_uri, OPT_NONE);
  $curl = curl_init($url);

  // Preflight request headers
  $header =
  [
    "origin: " . $url,
    "access-control-request-method: " . $request_method,
    "access-control-request-headers: " . $request_headers
  ];

  // Include response header
  check_curlopt(curl_setopt($curl, CURLOPT_HEADER, true));

  // Set common options
  check_curlopt(curl_setopt($curl, CURLOPT_HTTPHEADER, $header));
  check_curlopt(curl_setopt($curl, CURLOPT_RETURNTRANSFER, true));
  check_curlopt(curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $request_method));
  check_curlopt(curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true));

  $response = rest_api_curl_exec($curl);

  curl_close($curl);

  return $response;
}

// Get the tenant id of the current user
function rest_api_tenant_get_id()
{
  $response_json = rest_api_get("/auth");
  $info = json_decode($response_json);
  return $info->id_tenant;
}

// --------------------------------------------------------------
// ERROR FUNCTIONS
// --------------------------------------------------------------

// Returns last HTTP response code
function rest_api_get_http_response_code()
{
  global $_HTTP_RESPONSE_CODE;
  return $_HTTP_RESPONSE_CODE;
}

// Returns last HTTP response
function rest_api_get_http_response()
{
  global $_HTTP_RESPONSE;
  return $_HTTP_RESPONSE;
}

// Returns last cURL error number
function rest_api_get_curl_errno()
{
  global $_CURL_ERRNO;
  return $_CURL_ERRNO;
}

// Returns last cURL error message
function rest_api_get_curl_error()
{
  global $_CURL_ERROR;
  return $_CURL_ERROR;
}

// --------------------------------------------------------------
// QUERY HELPER FUNCTIONS
// --------------------------------------------------------------

// Returns cURL object
function rest_api_create_query(
  $request_method,
  $request_uri,
  $options = OPT_NONE,
  $data = null,
  $content_type = null)
{
  $url = rest_api_url($request_uri, $options);

  $curl = curl_init($url);

  // Query header entries
  $header = [];

  // Set auth header entry
  $auth_header = rest_api_auth_header($request_method, $request_uri);
  array_push($header, $auth_header);

  // Set content type header
  if (null !== $content_type)
  {
    array_push($header, "Content-type: " . $content_type);
  }

  // Set data for POST and PUT.
  if ($data !== null)
  {
    check_curlopt(curl_setopt($curl, CURLOPT_POSTFIELDS, $data));
  }

  // Common cURL options
  check_curlopt(curl_setopt($curl, CURLOPT_HTTPHEADER, $header));
  check_curlopt(curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $request_method));
  check_curlopt(curl_setopt($curl, CURLOPT_RETURNTRANSFER, true));
  check_curlopt(curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true));

  // Set header include/exclude options
  if ($options & OPT_HEADER_INCLUDE)
  {
    check_curlopt(curl_setopt($curl, CURLOPT_HEADER, true));
  }
  else
  if ($options & OPT_HEADER_ONLY)
  {
    check_curlopt(curl_setopt($curl, CURLOPT_HEADER, true));
    check_curlopt(curl_setopt($curl, CURLOPT_NOBODY, true));
  }

  return $curl;
}

function rest_api_url($request_uri, $options)
{
  global $_SERVER_URL;

  $url = $_SERVER_URL . $request_uri;
  if ($options & OPT_EXPAND)
  {
    $url .= "?expand";
  }

  return $url;
}

// Check the return value from curl_setopt
function check_curlopt($success)
{
  if (!$success)
  {
    echo "[rest_api_lib.php] Fatal Error: curl_setopt() failed".PHP_EOL;
    die();
  }
}

// Perform query and return response
// Return false on cURL error
// Use error functions to get error information
function rest_api_curl_exec($curl)
{
  global $_CURL_ERRNO;
  global $_CURL_ERROR;
  global $_HTTP_RESPONSE_CODE;
  global $_HTTP_RESPONSE;

  $_CURL_ERRNO         = 0;
  $_CURL_ERROR         = null;
  $_HTTP_RESPONSE_CODE = null;

  $response = curl_exec($curl);

  $http_response_code = curl_getinfo($curl, CURLINFO_RESPONSE_CODE);
  $_HTTP_RESPONSE_CODE = $http_response_code;
  $_HTTP_RESPONSE = $response;

  $errno = curl_errno($curl);
  if ($errno !== 0)
  {
    $error = curl_error($curl);
    $_CURL_ERRNO = $errno;
    $_CURL_ERROR = $error;

    return false;
  }

  return $response;
}

// Return auth header string
function rest_api_auth_header($request_method, $request_uri)
{
  global $_USERNAME;
  global $_PASSHASH;

  return rest_api_authorization_header(
    $_USERNAME,
    $_PASSHASH,
    $request_method,
    $request_uri);
}

// --------------------------------------------------------------
// RESPONSE PARSING FUNCTIONS
// --------------------------------------------------------------

// Get the header part of full response
function rest_api_response_header($response)
{
  $pos = strpos($response, "\r\n\r\n");
  if ($pos !== false)
  {
    $header = substr($response, 0, $pos);
  }
  else
  {
    $header = "";
  }

  return $header;
}

// Get the body part of full response
function rest_api_response_body($response)
{
  $pos = strpos($response, "\r\n\r\n");
  if ($pos !== false)
  {
    $header = substr($response, $pos + 4);
  }
  else
  {
    $body = "";
  }

  return $body;
}

// Returns reponse headers as a dictionary (array with keys and values)
function rest_api_parse_response_headers($response_header)
{
  $headers = [];

  $lines = explode("\n", $response_header);

  foreach($lines as $line)
  {
    $parts = explode(":", $line, 2);

    // Include header if $parts[1] exist
    if (isset($parts[1]))
    {
      $key = strtolower(trim($parts[0]));
      $value = trim($parts[1]);
      $headers[$key] = $value;
    }
  }

  return $headers;
}

// It is considered best practice to not close the <?php tag
// in a PHP-only file.