<?php
/*
File: updatefw.php

Update the application firmware of a device using the command line.

Assumes the device has one application microcontroller (script could be
updated to handle devices with multiple microcontrollers).

The version number used for the development application firmware is:

    199.199.199.x

The script will cycle between application firmware version
numbers ending with 200, 201, and 202. The previous development
firmware will be erased.

If the previous firmware is a non-development firmware (version
other that 199.199.199.x) it will not be erased.

For example, if the version of the current application firmware for
the device is:

    199.199.199.200

the script will update the version number to:

    199.199.199.201

If needed, you can alter this logic in the script below. For example,
different numbers could be used for the alternating firmware versions.

Command line usage:

    php updatefw.php LOGINFILE DEVICEID FIRMWAREFILE

Example invocation:

    php updatefw.php mylogin.json 42 /Users/miki/Documents/Arduino/Blink_Uno/Blink_Uno.hex

The above command will update the application firmware of the device
with id 42 using the .hex file given by the path in the last parameter.

Note on version numbers:

The version number must be unique for the microcontroller.
Suggested version number scheme is to use the first number as the
application identifier, followed by major/minor/build number:

    application number/id
    major version
    minor version
    build version (or "patch version")
*/

require_once __DIR__ . "/../lib/rest_api_lib.php";
require_once __DIR__ . "/../lib/device_lib.php";
require_once __DIR__ . "/../lib/ansi_codes.php";

// Call main function
main($argc, $argv);

function main($argc, $argv)
{
  // Set command params and read login config file
  if ($argc === 4)
  {
    $login_config_file     = $argv[1]; // absolute or relative path to config file
    $device_id             = $argv[2]; // id of device to update
    $firmware_file         = $argv[3]; // absolute or relative path of firmware file
    $microcontroller_index = 0;        // hardcoded value

    // Read login file
    $config_path = get_absolute_path($login_config_file);
    rest_api_read_login_config($config_path);
  }
  else
  {
    echo "USAGE:" . PHP_EOL;
    echo "php updatefw.php LOGINFILE DEVICEID FIRMWAREFILE" . PHP_EOL;
    echo "EXAMPLE:" . PHP_EOL;
    echo "php updatefw.php mylogin.json 42 Blink.ino.hex" . PHP_EOL;
    die();
  }

  echo "Querying device #{$device_id}" . PHP_EOL;

  // Get tenant id
  $tenant_id = rest_api_tenant_get_id();

  // Get device object
  $device_obj = device_get($tenant_id, $device_id);
  if (false === $device_obj)
  {
    echo "[ERROR] Could not find device with id: {$device_id}" . PHP_EOL;
    die();
  }

  // Get microcontroller object
  $microcontroller_obj = $device_obj->keys->microcontroller[$microcontroller_index];

  // Get current firmware name
  $firmware_name = $microcontroller_obj->firmware_appl;

  // Get firmware object
  $firmware_obj = firmware_appl_get($tenant_id, $firmware_name);
  if ((false === $firmware_obj) && ("factory" !== $firmware_name))
  {
    echo "[ERROR] Could not find firmware with name: {$firmware_name}" . PHP_EOL;
    die();
  }

  // Assume that the previous firmware should be deleted
  $firmware_delete_previous = true;

  if ("factory" === $firmware_name)
  {
    // Set dummy firmware version
    $firmware_version_array = explode(".", "0.0.0.0");
  }
  else
  {
    // Get firmware version number as an array
    $firmware_version_array = explode(".", $firmware_obj->info->version);
  }

  // Check if the current firmware is NOT a development firmware
  if (! is_development_firmware($firmware_version_array))
  {
    $firmware_version_array = explode(".", "199.199.199.0");
    $firmware_delete_previous = false;
  }

  // Get last version number entry (build number)
  $firmware_version_build = intval($firmware_version_array[3]);

  // Set alternating build number
  if (200 === $firmware_version_build)
  {
    $firmware_version_build_new = 201;
  }
  else
  if (201 === $firmware_version_build)
  {
    $firmware_version_build_new = 202;
  }
  else
  {
    $firmware_version_build_new = 200;
  }

  // Set new version number
  $firmware_version_array[3] = $firmware_version_build_new;
  $firmware_version_new = implode(".", $firmware_version_array);

  // Set new firmware name (use firmware file name)
  $firmware_name_new = "DEV_"  . $device_id . "_" .
    explode(".", basename($firmware_file))[0] . "_" .
    implode("_", $firmware_version_array);

  // Delete new development firmware if it already exists
  // It should not be in use
  $response = firmware_appl_get($tenant_id, $firmware_name_new);
  if (false !== $response)
  {
    // Firmware exists, attempt to delete it
    echo "Deleting existing application development firmware: " .
      $firmware_name_new . PHP_EOL;
    $success = firmware_appl_delete($tenant_id, $firmware_name_new);
    if (!$success)
    {
      echo "[ERROR] Could not delete application firmware: " .
        $firmware_name_new . PHP_EOL;
      echo "Check that it is not in use" . PHP_EOL;
    }
  }

  echo "Uploading new firmware: " . $firmware_name_new . PHP_EOL;
  sleep(1);

  // Upload new firmware file
  $firmware_absolute_path = get_absolute_path($firmware_file);
  $microcontroller_name = $microcontroller_obj->name;
  $firmware_description = $firmware_name_new;

  $response = firmware_appl_create(
    $tenant_id,
    $firmware_name_new,
    $microcontroller_name,
    $firmware_version_new,
    $firmware_description,
    $firmware_absolute_path);
  if (false === $response)
  {
    echo "[ERROR] Could not create firmware: " . $firmware_name_new . PHP_EOL;
    die();
  }

  echo "Updating device" . PHP_EOL;

  // Wait a little
  sleep(1);

  // Set new firmware of device microcontroller with the specified index
  $response = device_update_firmware_appl($tenant_id, $device_obj,
    $firmware_name_new, $microcontroller_index);
  if (false === $response)
  {
    echo "[ERROR] Failed to update application firmware of device id: {$device_id}" . PHP_EOL;
    die();
  }

  // Wait a little to ensure new device firmware is set on the server
  // (we cannot delete firmware in use)
  sleep(1);

  // Delete previous firmware
  if ($firmware_delete_previous)
  {
    echo "Deleting previous development firmware" . PHP_EOL;

    firmware_appl_delete($tenant_id, $firmware_name);
  }

  // Display a progress bar until next connection
  echo "Upload in progress... " . PHP_EOL;
  $connected = display_progress_bar($tenant_id, $device_id);

  if ($connected)
  {
      // Sleep a little at the end to give the illusion of work going on
      echo PHP_EOL . "Finalizing firmware update" . PHP_EOL;
      sleep(2);
  }
  else
  {
    echo "Device is offline" . PHP_EOL;
  }
}

function is_development_firmware($firmware_version_array)
{
  if (199 !== intval($firmware_version_array[0])) return false;
  if (199 !== intval($firmware_version_array[1])) return false;
  if (199 !== intval($firmware_version_array[2])) return false;
  return true;
}

// Display a progress bar while waiting for the device to connect
// (devices connect on a regular interval typically 60 seconds)
// Return true when estimated to be connect, false when device is offline
function display_progress_bar($tenant_id, $device_id)
{
  // Find sync period and time left to next connect
  $response = rest_api_get_expand("/global");
  $global_obj = json_decode($response);
  $interval = $global_obj->keys->sync_period_low;
  $time_last_seen = device_time_last_connect($tenant_id, $device_id);
  $time_estimated_next = $time_last_seen + $interval;

  // Display progress bar while waiting
  while (true)
  {
    $time_left = $time_estimated_next - time();

    if ($time_left >= 0)
    {
      // Show time left
      $time_elapsed = $interval - $time_left;
      CursorColumn(1);
      BackgroundWhite();
      echo str_repeat(" ", $time_elapsed);
      AnsiReset();
      echo str_repeat(" ", $time_left);
      echo VBAR;
      CursorColumn(1);
    }
    else
    if ($time_left < -5)
    {
      // Device is offline
      return false;
    }
    else
    {
      // Done
      return true;
    }

    sleep(1);
  }
}
